# AnyGateway

## Compilation

Project uses Qt6.

Set up out of tree build (recommended).
```shell
# mkdir build
# cd build
# qmake6 ../AnyGateway3.pro
# make
```

The binary is in `core/anygateway`. To run, create symlinks (or copy) all `*.so` files from the `plugin/` directory to `core/`.

# AnyGateway Config definition

The config file is a JSON file. The master element is an object containing objects for different aspects of the system.

The elements in the root object are:
- `core`: object that contains attributes for the core functionality.
- `format`: object that contains attributes needed to parse messages.
- `phonebook`: array that contains a list of known devices.
- `fetchbook`: object to configure phone book in SQL database.
- `deviceTypes`: object with infos about plugins (FIXME: no longer used?).
- `security`: object with infos about security features.
- `tools`: object with infos about certain internal tools; currently unused.

# First level objects

## core

The original idea of the software was that multiple cores may run on a system. This has never been implemented though, but the config file is still prepared for it. Thus another object called core is placed in the config that contains the actual config values:

- `id`: Name of the core section. Just put "core" there...
- `descr`: Description of the section. Ignored in the code, acts as comment.
- `enableConsole`: if true, the Admin Console is available
- `watchConfig`: if true, the config file is watched and automatically reloaded on change. Probably not completely reliable...
- `logging`: Object containing attributes to control logging.
  - `loglevel`: Can be "debug", "info", "warning", "error". Only log messages of the given level and higher are printed. Default: "info". (Moved from one level above).
  - `logFile`: Name (and path) of the logfile. Can be "stdout" or "stderr" to log to the respective output. Log file rotation is disabled in this case.
  - `rotateSize`: Size in bytes when the log file will be rotated. This value may be an integer, or a string with a suffix (k,M,G,T). Setting to 0 disables rotation. Minimal value: 10000.
  The program checks after writing each line whether the file size has exceeded this limit. So the resulting file sizes will not match exactly this value.
  - `pattern`: Regular expression to match log messages in log levels that are not currently logged. If the message matches, it is written to the log file. This can be used to log certain DEBG messages, even if log level DEBG is not activated.
- `pidFile`: Name of the File to write PID to. Set to "-" to disable. Program refuses to run if a valid PID file already exists.
- `numFiles`: If set, anygateway tries to increase its max number of open files. To do so successfully, `CAP_SYS_RESOURCE` is needed. Default value is usually 2048, so only about 2000 clients may connect at the same time (see `ulimit -a`).
- `plugins`: object containing the config for different plugins. A single plugin may be used multiple times, it then gets multiple objects here.

### plugins

This section contains the configurations for different plugins. Although the admin plugin is implemented as an input plugin, is is treated differently in the config file and is configured directly in the plugins section.

```
"plugins": {
  "descr": "List of all plugins",
  "pluginPath": ".",
  "Admin1": {
    ...
  },
  "input": {
     ...
  }
  "output": {
    ...
  }
}
```

The plugins are looked for in the path given by `pluginPath`. If it is in the `"plugins"` section, it is valid for all plugins. This value can be overridden with an attribute of the same name in every individual plugin.

If no path is given, the plugins are loaded from the directory the program was started in.

#### Admin plugin

```
"Admin1": {
    "enable": true,
    "classname": "AdminPlugin",
    "ipAddr": "127.0.0.1",
    "port": 6262,
    "timeout": 10000,
    "tries": 4,
    "password": "verysecret",
    "hash": "sha256",
    "passhash": "2bb80d537b1da3e38bd30361aa855686bde0eacd7162fef6a25fe97bf527a25b"
}
```

- `Admin1`: Name of the plugin. Must be different if multiple plugins are defined.
- `enable`: The plugin is only active if set to true.
- `descr`: Description string, ignored. Can be used as documentation.
- `classname`: Used to identify the dynamic library to load.
- `ipAddr`: The plugin only accepts connections on this IP address. From a security point of view, it makes sense to put "127.0.0.1" here; although this is not required. The name `listen` for this attribute also works. If an invalid IP address is provided, 127.0.0.1 is used.
- `interface`: Listen on the IP address of the interface with this name. Use `ipAddr` or `interface` or neither, but not both.
- `port`: listen on this port. Default is 6262.
- `timeout`: if nothing is received on the admin socket during this timeout (in seconds), the user is logged out. Default: 10s.
- `tries`: specify the number of password queries before the connection is closed again on entering the wrong password. Default: 3.
- `password`: Plain text password needed to log in.
- `hash`: defines the hash function that is used to protect the password (default: SHA256). Possible values: MD5, SHA, SHA1, SHA2, SHA224, SHA256, SHA384, SHA5212, SHA3, SHA3-224, SHA3-256, SHA3-384, SHA3-512
- `passhash`: hash value of the password. Currently unsalted.

For the input and output plugins, there are two more objects in this section called `"input": { ... }` and `"output": { ... }`.

#### Input Plugins

Currently available input plugins are (value for attribute `classname`):
- `UdpInputPlugin`. Single thread UDP handler.
- `TcpInputPlugin`. Single thread TCP handler.
- `TcpInputPlugin2`. Multi thread TCP handler, support for TLS.

```
"InputPlugin1": {
    "type": "input",
    "classname": "UdpInputPlugin",
    "port": "13179",
    "ipAddr": "172.16.10.1",
    "interface": "eth1",
    "autoClose": true,
    "timeout": 10000,
    "threadpoolSize": 8,
    "formatHandler": [
        "Handler1", "Handler2"
    ],
    "output": [
        "OutputPlugin1", "OutputPlugin3"
    ]
},
"InputPlugin2": {
    "type": "input",
    "classname": "TcpInputPlugin2",
    "port": 50024,
    "ipAddr": "172.16.20.2",
    "interface": "eth2",
    "autoClose": false,
    "timeout": 10000,
    "threadpoolSize": 4,
    "ssl": {
        "CAFile": "cacert.pem",
        "CACert": "MII...",
	"CertFile": "cert.pem",
	"Cert": "MII...",
	"KeyFile": "key.pem",
	"Key": "MII..."
    }
    "formatHandler": [
        "Handler2"
    ],
    "output": [
        "OutputPlugin2"
    ]
}
```

- `type`: Plugin type, must be set to "input".
- `descr`: Description string, ignored. Can be used as documentation.
- `classname`: Used to identify the shared library that contains the plugin.
- `ipAddr`: IP address to listen on for connections.
- `interface`: Listen on the IP address of the interface with this name. Use `ipAddr` or `interface` or neither, but not both.
- `port`: Port to listen on for connections.
- `autoClose`: if set to true, the connection is closed if no packets arrive during the specified interval.
- `timeout`: (tcp and tcp2) Timeout interval (in ms).
- `threadpoolSize`: (tcp2) number of threads that are kept around to process incoming data.
- `ssl`: (tcp2) if this section is present, SSL is used for encryption.
  - `verifyPeer`: if set to true (default), the peer must present a valid certificat for the connection to succeed. If set to false, the connection is still encrypted, but the Peer is not verified, i.e. there is no protection against MitM attacks.
  - `tlsVersion`: explicitely set TLS version to use. Possible values: 1.3, 1.3+, 1.2, 1.2+. The "+" means "or later".
  - `tlsDebug`: if set to yes, TLS handshake errors are logged and abort the connection
  - `CAFile`, `CertFile`, `KeyFile`: files containing the CACert, Certificate and Key, respectively. The `CAFile` is used to verify the peer certificate (if `verifyPeer` is set), the `CertFile` and `KeyFile` are presented to the peer upon request.
  - `CACertificate`, `Certificate`, `PrivateKey`: The certificates and key can be directly placed in the config file in PEM format. Use either files or PEM code for every element; different elements may have different input.
- `formatHandler`: Incoming messages are only passed to the format handlers defined here, other handlers will not see the data.
- `output`: send processed data (if event handler reports success) to these output plugins only.

#### Output Plugins

Available output plugins are (value for `classname`):
- `LoggerOutputPlugin`: write data to log file
- `UdpOutputPlugin`: send data via UDP
- `TcpOutputPlugin`: send data via TCP
- `SqlOutputPlugin`: write data to database
- `MqttOutputPlugin`: send message as MQTT to broker
- `TcpSrvOutputPlugin`: create TCP/SSL Server, send data as JSON

Every output plugin contains the attributes `type`: `output`, `descr`, and `classname` (function as for input plugins).

LoggerOutputPlugin:
- `file`: Name of the log file to write data to. May contain "~" for homedir.

TcpOutputPlugin, UdpOutputPlugin:
- `ipAddr`: where to send data to.
- `port`: destination port.
- `sourceIP`: set source IP address to this address (must be present on system).
- `sourceInterface`: set source IP address to address of this interface. Use only one of `sourceIP` or `sourceInterface`.
- `sourcePort`: set source port.
- `timeout`: (tcp) Connection is closed if no data is sent for this interval (in ms).

SqlOutputPlugin:
- `hostname`: name of SQL host.
- `port`: DB port (e.g. 3306 for MySQL).
- `db`: name of database to write to.
- `user`/`pw`: DB credentials.
- `threadpoolSize`: number of threads that are used to write data to DB.
- `msgperinsert`: aggregate up to this number of data sets per insert.
- `conntype`: Connection type ("ODBC").
- `ODBCName`: name of ODBC data source in system.
- `gpstable`: name of table to write GPS data to.
- `customtable`: name of table to write other data to.

MqttOutputPlugin:
- `target`: hostname of MQTT broker. Will be resolved using DNS.
- `ipAddr`: IP address of MQTT broker. Only used if `target` is empty.
- `port`: port to send MQTT to (1883).
- `username`: Username to use when connecting to broker.
- `password`: Password to use when connecting to broker.
- `topic`: array of strings that are concatenated to the topic (separated by "/"). If the name has the form "{{name}}", the field is replaced with the value of the variable with the given name. With the form "{{name:start:len}}", it is possible to extract a substring from the value. If len is not given, the value is used up to the end of the field. A maximum of 10 variables can be used in the topic.
- `clientid`: Set the clientid field of the MQTT message to this value. If not set, a randomly generated UUID will be used.
- `timeout`: Time in seconds the connection to the broker is kept open when no messages are sent.
- `ssl`: If defined, a MQTTs (TLS) connection is opened to the broker. Syntax is identical to the `ssl` section in the TcpInputPlugin2. Note: for the connection to succeed, the certificate of the broker must either contain the value of `target` as its CN, or the value of `ipAddr` as a SAN (type IP).

TcpSrvOutputPlugin:
- `ipAddr`: IP address for the TCP server to listen on
- `interface`: Interface for the TCP server to listen on. Use `ipAddr` or `interface` or neither, but not both.
- `port`: Port number to listen on.
- `ssl`: SSL configuration, identical to TcpInputPlugin2.
- `id`: Define which field of the Phone Book to use as ID in the JSON data. Possible values are
  - `Name`: Contents of the "name" field
  - `Type`: Contents of the "type" field (probably not very useful)
  - `IdType`: Contents of the "type" field in the "id" subobject
  - `PhoneNr`: Contents of the "PhoneNr" field
  - `Orig`: Use the ID that was transmitted by the client.
- `originalString`: JSON object defining whether the original data string should be included in the output data.
  - `enabled`: If set to true, the original data will be included.
  - `name`: The name given to the data containing original string.
- `timestamp`: JSON object defining whether a timestamp is included in the output data.
  - `enabled`, `name`: Same as for `originalString` (default value).
  - `timezone`: Set to "UTC" or "localtime" for timestamp in UTC or local time (default).
- `filter`: JSON object defining a filter to select which NMEA string are passed on. If no filters are defined, all NMEA strings are sent.
  - `field`: Name of a field of the original data to apply filter to.
  - `value`: If the field contains this value, the NMEA message is sent.
- `keepAlive`: If set to a value n > 0, the plugin will send a keepalive message every n seconds if no other data has been sent. If set to 0, no keepalives will be sent. If set to a value n < 0, a keep alive will be sent every n seconds regardless whether other data has been sent.
- `keepAliveTime`: If set to true, the keep alive message will contain an additional field "interval" containing the value of keepAlive.
- `compact`: If set to true, the JSON data will be in compact form (no additional white space), otherwise in human readable form.
- `framing`: JSON object containing the framing characters or strings to separate JSON messages on the ouptput. If not set, no framing chars will be used.
  - `prefix`: If this is a string, it is literally taken as framing prefix. To use non printable characters, this can be an array of strings containing the hex values of the framing prefix bytes. E.g. prefix:  [ "02" ] for STX.
  - `postfix`: Same as prefix. E.g. postfix: [ "03" ] for ETX.

The plugin creates a JSON object containing the following data:
```
{
    "NMEA": {
        
    },
    "Timestamp": "[current time]",
    "ID": "[device ID]",
    "originalString": "[original data]"
}
```

In the NMEA block, all fields from the NMEA string are included, except those named "ignore". This value can be set using the format manager (see below).

## format

This section defines formats for input strings that must be parsed.

- internal: probably not used anymore.
- types: object containing the different strings.
- handler: configure handlers for different input strings.

```
"format": {
    "internal": {

    },
    "types": {

    },
    "handler": {

    }
}
```

### Format types
Specify the format of the input strings that will be passed to the plugin.
These format types are available, specified in the `type` field:
- `simple`
- `nmea`
- `json`
Common attributes:
- `recognitionPattern`: Regular expression that will be applied to the input stream to find messages of this type. The regexp must match the complete string, as the matched string is then further processes. For `nmea`, the pattern `"\$G[PN][^\\*]*\\*.." may be used. For the `json` format, the regexp might look like `"^{\".*\":.*}$"`.

The format `simple` can be used to match any type of strings, which are defined with the following attributes:
- `startChar`: First character of the string.
- `endChar`: Last character of the string.
- `separator`: Character used to separate data fields in the string.
- `parsedDataKey`: FIXME ???
- `format`: FIXME ???
```
"Simple": {
    "type": "simple",
    "recognitionPattern": "^{.*}$",
    "startChar": "{",
    "endChar": "}",
    "separator": ",",
    "parsedDataKey": "AnyWebData"
}
```

The format `nmea` can be used to parse data incoming as NMEA strings. If `checkChecksum` is set to true, the NMEA checksum is checked, and strings with an incorrect checksum are ignored.
The following NMEA strings are recognized, including the default names of the data fields (the first field "type" will contain "$GPGGA", "$GPRMC", ...):
- `GPGGA`: type, hhmmss, latitude, ns, longitude, ew, fs, nosv, hdop, msl, mslm, altref, altrefm, diffage, diffstation
- `GPRMC`: type, hhmmss, status, latitude, ns, longitude, ew, spd, dir, ddmmyy, mv, mve [, modind {>=NMEA 2.3}]
- `GPGLL`: type, latitude, ns, longitude, ew, hhmmss, valid [, modind {NMEA >=3.0}]
- `GPGSA`: modema, mod123, prn1gps, prn1sbas, prn1glo, prn2gps, prn2sbas, prn2glo, prn3gps, prn3sbas, prn3glo, prn4gps, prn4sbas, prn4glo, pdop, hdop, vdop, sysid
- `GPGSV`: type, nomsg, msgno, nosv, sv, elv, az, cno
- `GPVTG`: type, cogt, t, cogm, m, sog, n, kph, k, [, modind {NMEA >=2.3}]
- `GPTXT`
For every string, a separate Object can be configured to specify the field names for further processing. These names are relevant e.g. in the SQL or TcpSrv Ouptut Plugins. A field named "id" is used to match the client sending that string to entries in the phone book.

For date, time and coordinate fields, automatic format conversion can be used by choosing the field name appropriately. If the field name begins with "latitude" or "longitude", the coordinates are converted from the NMEA format "ddmmm.mmmmm" to the decimal "dd.ddddd" format. If the field name begins with "time", the it converts from "HHMMSS[.xx]" to "HH:MM:SS". The field name "ddmmyy" converts from "ddmmyy" to "yyyy-mm-dd". For these fields, the new name of the field can be added after a colon.

Example: Assume the latitude value is "4724.1234". If the field name is specified as "latitude": "Latitude", the data will contain "Latitude = 4724.1234". If specified as "latitude": "latitude:Latitude", the data will be "Latitude = 47.4020567".

For the TcpSrv Output Plugin, set field names to "ignore" to prevent them from being included in the output. Multiple fields can be named "ignore".
```
"GPGGA": { "type": "type", "hhmmss": "MsgTime", "latitude": "Lat", [...]}
```

For `GPTXT` strings, no default field names are defined. The first field after "GPTXT" is the subid that distinguish further strings. Field names can be defined as:
```
"GPTXT": {
  "ID": "type,msgtype,idtype,id",
  "hostname": "type,custval1,custval2,id"
}
```
If the default field names are suitable, only the object for "GPTXT" is needed, the configs for all other "GPXXX" messages can be omitted.

```
"NMEA": {
    "type": "nmea",
    "recognitionPattern": "^\\$G[PN][^\\*]*\\*..",
    "GPTXT": {
      "ID": "type,custkeyname,id",
      "CAN": "type,custkeyname,custsubkeyname,custsubkeyvalue,custvalue1,custvalue2"
    }
}
```
The format recognizes this attributes:

The format `json` is used to parse incoming JSON messages.
```
"JSON": {
    "version": 0.2,
    "recognitionPattern": "^\{\".*\":.*}$"
}
```

### Format handler
Specify format handlers to parse incoming data. The name of the object must be the name of the handler:

 ```
"handler": {
  "SimpleInHandler": {
    "handles": [ "Simple" ]
  } 
  "NMEAInHandler": {
    "handles": [ "NMEA" ]
  },
  "JSONInHandler": {
    "handles": [ "JSON" ]
  }
}
 ```

## phonebook

All devices that want to send data to the AnyGateway must be registered in the phonebook for the messages to be processed.

### Static phonebook
The phonebook is an array of objects defining the devices and the ID fields. Every entry must have the attributes `enabled` (a boolean value) and `id` (an non-empty array). The following example shows all attributes that are evaluated.

```
{
    "name": "Device1",
    "enabled": true,
    "type": "AnyRover",
    "phoneNr": "0790123456",
    "id": [
      {
        "type": "IMSI",
        "id": "22801123456789"
      }
    ]
}
```
The inner `type` field can be used to specify the type of match for the ID:
- `prefix`: Match if the device ID begins with the specified string.
- `postfix`: Match if the device ID ends with the specified string.
- `contains`: Match if the device ID contains the specified string. Any `prefix` or `postfix` match also matches a `contains` condition.
- `regexp`: Match if the devices ID matches the given regular expression. Example: The object
`"id": [{"type": "regexp", "id": "ECAB102347[0-9]{4}"}]` matches the IDs `ECAB1023470000` - `ECAB1023479999`, but not `ECAB102347000`.
- For any other type, a simple string comparison is performed, and the values must match exactly.

### Dynamic phonebook
The phonebook can be placed into an SQL database, where it is frequently checked and updated. The access to the SQL database itself is defined in the config file. The fetchbook object is placed in the root of main config object.

The config file still must contain a phonebook object, even if this can be empty. Due to the design of the code, the fetchbook is not activated if the phonebook object is missing.

```
"phonebook": [
],
"fetchbook": {
    "enabled": true,
    "database": {
        "conntype": "ODBC",
        "hostname": "db.example.org",
        "port": 3306,
        "db": "phonebook",
        "user": "admin",
        "pw": "verysecret",
        "ODBCName": "ODBCPhonebook",
        "table": "phoneusers"
    },
    "fields": {
        "json": {
            "devname": "name",
            "phnumber": "phoneNr",
            "type": "type",
            "enabled": "true",
            "id_type": "type",
            "id_value": "imsi"
        },
	"sql": {
	    "devname": "device_name",
	    "phnumber": "phone_number",
	    "imsi": "imsi"
	}
    },
    "interval": 30,
    "minrecords": 5
}
```

The value of `conntype` defines, which other attributes are evaluated for database access.
- conntype `MYSQL` to access a MySQL or MariaDB database:
  - `hostname`: the server host name
  - `port`: the server port (3306)
  - `user`: username of the MySQL user
  - `pw`: password of the MySQL user
- conntype `ODBC` to access ODBC registered database
  - `odbcname`: name the database is registered as

The value `table` is used in either configuration to designate the table to read the data from.

The `fields` object defines the names of the fields for the phonebook objects. For an SQL database access, the columns of the table are defined in the `sql` object:
- `devname`: Name of the column containing the device name
- `phnumber`: Name of the column containing the phone number.
- `imsi`: Name of the column containing the IMSI field.
- `enabled`: Name of the column containing the enabled field.

The database table looks like this, the value in the column `imsi` is used as ID. Here the default names of the column names are given.
```
show create table fields;
CREATE TABLE `fields` (
  `device_name` varchar(64) NOT NULL,
  `phone_number` varchar(32) DEFAULT NULL,
  `imsi` varchar(32) DEFAULT NULL,
  `enabled` boolean,
  PRIMARY KEY (`device_name`)
);
```
The column `enabled` is optional and can also be of another type. Values like 0, "" (empty string), "no", "NO", "false", "FALSE" are interpreted as boolean value false, everything else as true.

When reading the SQL data, the server inserts JSON objects into the phonebook according to this schema, where the names of the JSON attributes can be defined in the `json` object. In this example, the default names of the attributes are used.
```
{
  "enabled": true,
  "id": [
    {
      "id": "12345",
      "type": "imsi"
    }
  ],
  "name": "Device001",
  "phoneNr": "0790123456",
  "type": "GENERATED"
}
```

The attributes are set to these values:
- `enabled` is set to true or the value from `enabled` if defined.
- `type` is always set to "GENERATED"
- `name` is set to the value from `devname`
- `phoneNr` is set to the value from `phnumber`
- `type` is set to "imsi"
- `id` (inner) is set to the value from `imsi`

The value `interval` defines, how often (in minutes) the table in the database is queried for new entries. If less than `minrecords` are read from the database, the data is ignored. This is to save the day on unexpected table flushes or database read errors...

The server will start with the phonebook loaded from the phonebook object. After the interval has expired, it will read the data from the SQL database, thereby discarding the current phone book. After a successful read, the server will overwrite the current config file, keeping everything except the phonebook as is. So on the next run, it will start with the current phonebook. The old config file is stored with a timestamp added to the file name.

This procedure is repeated whenever the interval timeout expires. If the current phonebook and the database contain the same data, no new config file is written.

Note: since JSON does not guarantee the order of attributes in an object, the newly written config file may be reordered in respect to the original file. Only arrays keep the order of their entries.

## tools

In this section, parameters for external tools may be specified. Currently not used, as no external tools are called. There used to be a command to send SMS via Email, whose attributes are left in place as reference.

```
"tools": {
    "sendsms": {
        "mail_server": "",
        "mail_from": "",
        "mail_to": "",
        "mail_subject": ""
    }
}
```
